<?php

/**
 * @file
 * Contains the administrative functions of the print UI module.
 *
 * This file is included by the print UI module, and includes the
 * settings form.
 *
 * @ingroup print
 */

/**
 * Form constructor for the send by email module settings form.
 *
 * @ingroup forms
 */
function print_ui_settings() {
  $form['settings'] = array(
    '#type' => 'fieldset',
    '#title' => t('Link options'),
  );

  foreach (module_implements('print_link') as $module) {
    $function = $module . '_print_link';
    if (function_exists($function)) {
      $link = call_user_func_array($function, array());
      $format = $link['format'];

      $form['settings']['checkboxes']['link_pos']['print_' . $format . '_link_pos'] = array(
        '#type' => 'checkboxes',
        '#title' => t('Link location'),
        '#default_value' => variable_get('print_' . $format . '_link_pos', drupal_json_decode(PRINT_UI_LINK_POS_DEFAULT)),
        '#options' => array('link' => t('Links area'), 'corner' => t('Content corner'), 'block' => t('Block'), 'help' => t('Help area')),
        '#description' => t('Choose the location of the link(s) to the printer-friendly version pages. The Links area is usually below the node content, whereas the Content corner is placed in the upper-right corner of the node content (these apply only in node pages). The Help area is used for system pages. Enabling the block area link will allow links of this format in the provided links module. Unselect all options to disable the link. Even if the link is disabled, you can still go to %path where nid is the numeric id of the node.', array('%path' => $link['path'] . '/nid')),
      );

      $form['settings']['checkboxes']['link_teaser']['print_' . $format . '_link_teaser'] = array(
        '#type' => 'checkbox',
        '#title' => t('Display link in teaser'),
        '#default_value' => variable_get('print_' . $format . '_link_teaser', PRINT_UI_LINK_TEASER_DEFAULT),
        '#description' => t('Enabling this will display the link in teaser mode.'),
      );
    }
  }

  $form['settings']['adv_link'] = array(
    '#type' => 'fieldset',
    '#title' => t('Advanced link options'),
    '#collapsible' => TRUE,
    '#collapsed' => FALSE,
  );

  foreach (module_implements('print_link') as $module) {
    $function = $module . '_print_link';
    if (function_exists($function)) {
      $link = call_user_func_array($function, array());
      $format = $link['format'];

      $form['settings']['adv_link']['show_link']['print_' . $format . '_show_link'] = array(
        '#type' => 'radios',
        '#title' => t('Link style'),
        '#default_value' => variable_get('print_' . $format . '_show_link', PRINT_UI_SHOW_LINK_DEFAULT),
        '#options' => array(1 => t('Text only'), 2 => t('Icon only'), 3 => t('Icon and Text')),
        '#description' => t('Select the visual style of the link.'),
      );

      $form['settings']['adv_link']['link_use_alias']['print_' . $format . '_link_use_alias'] = array(
        '#type' => 'checkbox',
        '#title' => t('Use URL alias instead of node ID'),
        '#default_value' => variable_get('print_' . $format . '_link_use_alias', PRINT_UI_LINK_USE_ALIAS_DEFAULT),
        '#description' => t('Enabling this will create the link using the URL alias instead of the node ID.'),
      );

      $form['settings']['adv_link']['link_class']['print_' . $format . '_link_class'] = array(
        '#type' => 'textfield',
        '#title' => t('Link class'),
        '#default_value' => variable_get('print_' . $format . '_link_class', $link['class']),
        '#size' => 20,
        '#maxlength' => 250,
        '#description' => t('This can be used by themers to change the link style or by jQuery modules to open in a new window (e.g. greybox or thickbox). Multiple classes can be specified, separated by spaces.'),
      );

      $form['settings']['adv_link']['sys_link_visibility']['print_' . $format . '_sys_link_visibility'] = array(
        '#type' => 'radios',
        '#title' => t('Show link in system (non-content) pages'),
        '#description' => 'Any page that is not a Drupal node. Usually pages generated by Drupal or a module such as Views or Panels.',
        '#default_value' => variable_get('print_' . $format . '_sys_link_visibility', PRINT_UI_SYS_LINK_VISIBILITY_DEFAULT),
        '#options' => array(t('Show on every page except the listed pages.'), t('Show on only the listed pages.')),
      );

      $form['settings']['adv_link']['sys_link_pages']['print_' . $format . '_sys_link_pages'] = array(
        '#type' => 'textarea',
        '#default_value' => variable_get('print_' . $format . '_sys_link_pages', PRINT_UI_SYS_LINK_PAGES_DEFAULT),
        '#rows' => 3,
        '#description' => t('Setting this option will add a link on pages created by Drupal or the enabled modules.') . '<br />' .
                          t("Enter one page per line as Drupal paths. The '*' character is a wildcard. Example paths are %blog for the blog page and %blog-wildcard for every personal blog. %front is the front page.", array('%blog' => 'blog', '%blog-wildcard' => 'blog/*', '%front' => '<front>')),
      );

      if (module_exists('php')) {
        $access = user_access('use PHP for settings');

        if ($form['settings']['adv_link']['sys_link_visibility']['print_' . $format . '_sys_link_visibility']['#default_value'] == 2 && !$access) {
          $form['settings']['adv_link']['sys_link_visibility']['print_' . $format . '_sys_link_visibility'] = array('#type' => 'value', '#value' => 2);
          $form['settings']['adv_link']['sys_link_pages']['print_' . $format . '_sys_link_pages'] = array('#type' => 'value', '#value' => $form['settings']['adv_link']['print_' . $format . '_sys_link_pages']['#default_value']);
        }
        elseif ($access) {
          $form['settings']['adv_link']['sys_link_visibility']['print_' . $format . '_sys_link_visibility']['#options'][] = t('Show if the following PHP code returns <code>TRUE</code> (PHP-mode, experts only).');
          $form['settings']['adv_link']['sys_link_pages']['print_' . $format . '_sys_link_pages']['#description'] .= ' ' . t('If the PHP-mode is chosen, enter PHP code between %php. Note that executing incorrect PHP-code can break your Drupal site.', array('%php' => '<?php ?>'));
        }
      }

      $form['settings']['adv_link']['book_link']['print_' . $format . '_book_link'] = array(
        '#type' => 'radios',
        '#title' => t('Link in book hierarchy nodes'),
        '#default_value' => variable_get('print_' . $format . '_book_link', PRINT_UI_BOOK_LINK_DEFAULT),
        '#options' => array(t('No link'), t('Current page and sub-pages'), t('Current page only')),
      );
    }
  }

  $form['#theme'] = 'print_ui_settings';

  return system_settings_form($form);
}

/**
 * Returns HTML for the link settings page.
 *
 * @param array $variables
 *   An associative array containing:
 *   - form: A render element representing the form.
 *
 * @ingroup themeable
 * @ingroup print_themeable
 */
function theme_print_ui_settings($variables) {
  $form = $variables['form'];
  $header = array('');

  foreach (module_implements('print_link') as $module) {
    $function = $module . '_print_link';
    if (function_exists($function)) {
      $link = call_user_func_array($function, array());

      $header[] = array('data' => $link['format'], 'class' => 'checkbox');
    }
  }

  foreach (element_children($form['settings']) as $group) {
    $rows = array();
    foreach (element_children($form['settings'][$group]) as $elem) {
      $row = array();

      foreach (module_implements('print_link') as $module) {
        $function = $module . '_print_link';
        if (function_exists($function)) {
          $link = call_user_func_array($function, array());
          $format = $link['format'];

          $item = $form['settings'][$group][$elem]['print_' . $format . '_' . $elem];

          if (empty($row)) {
            // Start row, fill title and description
            $row_desc = array(
              '#type' => 'item',
              '#markup' => isset($item['#title']) ? $item['#title'] : '',
              '#description' => isset($item['#description']) ? $item['#description'] : '',
            );

            $row[] = array('data' => drupal_render($row_desc), 'class' => array(''));
          }

          $item['#title_display'] = 'invisible';
          $item['#description'] = '';

          $class = ($item['#type'] == 'checkbox') ? 'checkbox' : '';

          $row[] = array('data' => drupal_render($item), 'class' => array($class));
        }
      }
      $rows[] = $row;
    }

    $form['settings'][$group]['#children'] = theme('table', array('header' => $header, 'rows' => $rows, 'attributes' => array('id' => 'permissions')));
  }

  $output = drupal_render_children($form);

  return $output;
}
